"""https://github.com/kuangliu/pytorch-cifar/blob/master/models/resnet.py"""
import torch
import torch.nn as nn
import torch.nn.functional as F


class BasicBlock(nn.Module):
    expansion = 1

    def __init__(self, in_planes, planes, stride=1):
        super(BasicBlock, self).__init__()
        self.conv1 = nn.Conv2d(
            in_planes, planes, kernel_size=3, stride=stride, padding=1, bias=False
        )
        self.bn1 = nn.BatchNorm2d(planes)
        self.conv2 = nn.Conv2d(
            planes, planes, kernel_size=3, stride=1, padding=1, bias=False
        )
        self.bn2 = nn.BatchNorm2d(planes)

        self.shortcut = nn.Sequential()
        if stride != 1 or in_planes != self.expansion * planes:
            self.shortcut = nn.Sequential(
                nn.Conv2d(
                    in_planes,
                    self.expansion * planes,
                    kernel_size=1,
                    stride=stride,
                    bias=False,
                ),
                nn.BatchNorm2d(self.expansion * planes),
            )

    def forward(self, x):
        out = F.relu(self.bn1(self.conv1(x)))
        out = self.bn2(self.conv2(out))
        out += self.shortcut(x)
        out = F.relu(out)
        return out


class Bottleneck(nn.Module):
    expansion = 4

    def __init__(self, in_planes, planes, stride=1):
        super(Bottleneck, self).__init__()
        self.conv1 = nn.Conv2d(in_planes, planes, kernel_size=1, bias=False)
        self.bn1 = nn.BatchNorm2d(planes)
        self.conv2 = nn.Conv2d(
            planes, planes, kernel_size=3, stride=stride, padding=1, bias=False
        )
        self.bn2 = nn.BatchNorm2d(planes)
        self.conv3 = nn.Conv2d(
            planes, self.expansion * planes, kernel_size=1, bias=False
        )
        self.bn3 = nn.BatchNorm2d(self.expansion * planes)

        self.shortcut = nn.Sequential()
        if stride != 1 or in_planes != self.expansion * planes:
            self.shortcut = nn.Sequential(
                nn.Conv2d(
                    in_planes,
                    self.expansion * planes,
                    kernel_size=1,
                    stride=stride,
                    bias=False,
                ),
                nn.BatchNorm2d(self.expansion * planes),
            )

    def forward(self, x):
        out = F.relu(self.bn1(self.conv1(x)))
        out = F.relu(self.bn2(self.conv2(out)))
        out = self.bn3(self.conv3(out))
        out += self.shortcut(x)
        out = F.relu(out)
        return out


class ResNet(nn.Module):
    def __init__(self, block, num_blocks, width=1, num_classes=10):
        super(ResNet, self).__init__()
        self.in_planes = 64
        self.width = width

        self.conv1 = nn.Conv2d(3, 64, kernel_size=3, stride=1, padding=1, bias=False)
        self.bn1 = nn.BatchNorm2d(64)
        self.layer1 = self._make_layer(block, 64 * self.width, num_blocks[0], stride=1)
        self.layer2 = self._make_layer(block, 128 * self.width, num_blocks[1], stride=2)
        self.layer3 = self._make_layer(block, 256 * self.width, num_blocks[2], stride=2)
        self.layer4 = self._make_layer(block, 512 * self.width, num_blocks[3], stride=2)
        self.linear = nn.Linear(512 * block.expansion * self.width, num_classes)

    def _make_layer(self, block, planes, num_blocks, stride):
        strides = [stride] + [1] * (num_blocks - 1)
        layers = []
        for stride in strides:
            layers.append(block(self.in_planes, planes, stride))
            self.in_planes = planes * block.expansion
        return nn.Sequential(*layers)

    def forward(self, x):
        out = F.relu(self.bn1(self.conv1(x)))
        out = self.layer1(out)
        out = self.layer2(out)
        out = self.layer3(out)
        out = self.layer4(out)
        out = F.avg_pool2d(out, 4)
        out = out.view(out.size(0), -1)
        out = self.linear(out)
        return out

    def get_intermediate(self, x):
        lookup = {}
        out = F.relu(self.bn1(self.conv1(x)))
        lookup["preprocess"] = out
        out = self.layer1(out)
        lookup["layer1"] = out
        out = self.layer2(out)
        lookup["layer2"] = out
        out = self.layer3(out)
        lookup["layer3"] = out
        out = self.layer4(out)
        lookup["layer4"] = out
        out = F.avg_pool2d(out, 4)
        out = out.view(out.size(0), -1)
        lookup["avg_pool"] = out
        out = self.linear(out)
        lookup["logit"] = out
        return lookup

    @property
    def available_repr(self):
        return [
            "preprocess",
            "layer1",
            "layer2",
            "layer3",
            "layer4",
            "avg_pool",
        ]

    def combine_model(self, model_1, model_2, breakpoint=-1):
        if breakpoint == -1 or breakpoint <= 5:
            model_1.linear = model_2.linear
        if breakpoint <= 4:
            model_1.layer4 = model_2.layer4
        if breakpoint <= 3:
            model_1.layer3 = model_2.layer3
        if breakpoint <= 2:
            model_1.layer2 = model_2.layer2


class ResNetSimCLR(nn.Module):
    def __init__(self, base_model, out_dim):
        super(ResNetSimCLR, self).__init__()

        self.backbone = base_model
        dim_mlp = self.backbone.linear.in_features

        # add mlp projection head
        self.backbone.linear = nn.Sequential(
            nn.Linear(dim_mlp, dim_mlp), nn.ReLU(), self.backbone.linear
        )

    def forward(self, x):
        return self.backbone(x)

    def get_intermediate(self, x):
        return self.backbone.get_intermediate(x)


def ResNet18():
    return ResNet(BasicBlock, [2, 2, 2, 2])


def ResNet34():
    return ResNet(BasicBlock, [3, 4, 6, 3])


def ResNet50():
    return ResNet(Bottleneck, [3, 4, 6, 3])


def ResNet101():
    return ResNet(Bottleneck, [3, 4, 23, 3])


def ResNet152():
    return ResNet(Bottleneck, [3, 8, 36, 3])
